﻿#include "precompiled.h"
#include "common.h"
#include "Direct3DBase.h"

#include "AppWindow.h"
#include "CommonStates.h"

using namespace DirectX;

#ifdef ENABLE_DIRECT2D
#pragma comment(lib,"d2d1.lib")
#pragma comment(lib, "dwrite.lib") 
#endif

namespace RTCam {

static const wchar_t DefaultFontName[] = L"Verdana";
static const float DefaultFontSize = 18;
const float Direct3DBase::RecommendedLineSpacing = DefaultFontSize * 1.2f;

Direct3DBase::Direct3DBase(void) :
	m_d3dDevice(nullptr),
	m_deviceContext(nullptr),
	m_swapChain(nullptr),
	m_userAnnotations(nullptr),
	// DirectX Helpers
	m_states(nullptr),
	m_spriteBatch(nullptr),
	// Other stuff
	m_finalRenderTargetView(nullptr),
	m_finalDepthStencilView(nullptr),
	m_featureLevel((D3D_FEATURE_LEVEL)0),
	m_renderSize(0, 0),
	m_window()
{
}


Direct3DBase::~Direct3DBase(void)
{
}

void Direct3DBase::Initialize(const shared_ptr<AppWindow>& window)
{
	ASSERT_MSG(window != nullptr, "A null AppWindow pointer was passed to Direct3DBase::Initialize");
	m_window = window;

	CreateDeviceResources();
	CreateWindowSizeDependentResources();
}

void Direct3DBase::HandleDeviceLost()
{
	// Reset these member variables to ensure that UpdateForWindowSizeChange recreates all resources.
	m_renderSize.x = 0;
	m_renderSize.y = 0;
	m_swapChain = nullptr;

	CreateDeviceResources();
	UpdateForWindowSizeChange();
}

// These are the resources that depend on the device.
void Direct3DBase::CreateDeviceResources()
{
	// This flag adds support for surfaces with a different color channel ordering
	// than the API default. It is required for compatibility with Direct2D.
	UINT creationFlags = D3D11_CREATE_DEVICE_BGRA_SUPPORT;

#if defined(_DEBUG)
	// If the project is in a debug build, enable debugging via SDK Layers with this flag.
	creationFlags |= D3D11_CREATE_DEVICE_DEBUG;
#endif

	// This array defines the set of DirectX hardware feature levels this app will support.
	// Note the ordering should be preserved.
	D3D_FEATURE_LEVEL featureLevels[] = {
		D3D_FEATURE_LEVEL_11_1,
		D3D_FEATURE_LEVEL_11_0,
		
		// TODO: D3D10 should work if the shaders are compiled for HLSL 4.
		//D3D_FEATURE_LEVEL_10_1,
		//D3D_FEATURE_LEVEL_10_0,
	};
	UINT numFeatureLevels = ARRAYSIZE(featureLevels);

	// Create the Direct3D 11 API device object and a corresponding context.
	ComPtr<ID3D11Device> device;
	ComPtr<ID3D11DeviceContext> context;
	ThrowIfFailed(
		D3D11CreateDevice(
			nullptr,					// Specify nullptr to use the default adapter.
			D3D_DRIVER_TYPE_HARDWARE,	
			nullptr,					// Unused (Not using a software rasterizer)
			creationFlags,				// Set debug and Direct2D compatibility flags.
			featureLevels,				// List of feature levels this app can support.
			ARRAYSIZE(featureLevels),
			D3D11_SDK_VERSION,
			&device,					// Returns the Direct3D device created.
			&m_featureLevel,			// Returns feature level of device created.
			&context					// Returns the device immediate context.
		)
	);

	// Get the Direct3D 11.1 API device and context interfaces.
	ThrowIfFailed(device.As(&m_d3dDevice));
	ThrowIfFailed(context.As(&m_deviceContext));
	ThrowIfFailed(context.As(&m_userAnnotations));

	string deviceName = "D3DBase:D3DDevice";
	m_d3dDevice->SetPrivateData(WKPDID_D3DDebugObjectName, deviceName.length(), deviceName.c_str());
	SetDebugObjectName(m_deviceContext.Get(), "D3DBase:DeviceContext");
	
	// Initialize the DirectX helper classes
	m_states = unique_ptr<CommonStates>(new CommonStates(device.Get()));
	m_spriteBatch = unique_ptr<SpriteBatch>(new SpriteBatch(context.Get()));

	// Create the Direct2D factory.
#ifdef ENABLE_DIRECT2D
	ThrowIfFailed(
		D2D1CreateFactory(D2D1_FACTORY_TYPE_SINGLE_THREADED, m_d2dFactory.GetAddressOf())
	);

	// Create the DirectWrite factory.
	ThrowIfFailed(
		DWriteCreateFactory(
			DWRITE_FACTORY_TYPE_SHARED,
			__uuidof(m_dWriteFactory),
			reinterpret_cast<IUnknown **>(m_dWriteFactory.GetAddressOf())
		)
	);

	// Create the text format object.
	ThrowIfFailed(
		m_dWriteFactory->CreateTextFormat(
			DefaultFontName,
			nullptr,
			DWRITE_FONT_WEIGHT_NORMAL,
			DWRITE_FONT_STYLE_NORMAL,
			DWRITE_FONT_STRETCH_NORMAL,
			DefaultFontSize,
			L"", // Locale
			m_defaultTextFormat.GetAddressOf()
		)
	);
#endif

	m_userAnnotations->SetMarker(L"D3DBase: Device resources created.");
}

void Direct3DBase::CreateWindowSizeDependentResources()
{
	m_userAnnotations->BeginEvent(L"D3DBase: Create Window Size Dependent Resources");

	ASSERT(!m_window.expired());

	// Store the window bounds so the next time we get a SizeChanged event we can
	// avoid rebuilding everything if the size is identical.
	m_renderSize = m_window.lock()->GetClientSize();

	// The color format for the swap chain to be created
	// NOTE: Because of the relaxed render target creation rules that Direct3D 11 has for
	// back buffers, applications can create a DXGI_FORMAT_B8G8R8A8_UNORM_SRGB render
	// target view from a DXGI_FORMAT_B8G8R8A8_UNORM swap chain so they can use automatic
	// color space conversion when they render the swap chain.
	const auto swapChainFormat = DXGI_FORMAT_B8G8R8A8_UNORM;
	const auto renderTargetViewFormat = DXGI_FORMAT_B8G8R8A8_UNORM_SRGB;

	const auto dsFormat = DXGI_FORMAT_D24_UNORM_S8_UINT;
	const auto dsvFormat = dsFormat;
	const int dsBindFlags = D3D11_BIND_DEPTH_STENCIL;

	const int bufferCount = 2; // Use a double-buffered swap chain.

	// Match the window size
	int bufferWidth = m_renderSize.x;
	int bufferHeight = m_renderSize.y;

	if(m_swapChain != nullptr) {
		// If the swap chain already exists, resize it.

		// Resize the buffers
		ThrowIfFailed(m_swapChain->ResizeBuffers(bufferCount, bufferWidth, bufferHeight, swapChainFormat, 0));

		// Recreate the D2D render target and text brush
		CreateD2DWindowSizeDependentResources();

	} else {
		// Otherwise, create a new one using the same adapter as the existing Direct3D device.

		// First make sure any old swap chains are fully destroyed.
		m_deviceContext->ClearState();
		m_deviceContext->Flush();

		// The description for the swap chain to be created.
		DXGI_SWAP_CHAIN_DESC1 scDesc = {0};
		scDesc.Width = bufferWidth;
		scDesc.Height = bufferHeight;
		scDesc.Format = swapChainFormat;
		scDesc.Stereo = false;
		scDesc.SampleDesc.Count = 1;	// No multi-sampling
		scDesc.SampleDesc.Quality = 0;
		scDesc.BufferUsage = DXGI_USAGE_RENDER_TARGET_OUTPUT;
		scDesc.BufferCount = bufferCount;
		scDesc.Scaling = DXGI_SCALING_NONE;
		scDesc.SwapEffect = DXGI_SWAP_EFFECT_FLIP_SEQUENTIAL;
		//scDesc.Scaling = DXGI_SCALING_STRETCH;
		//scDesc.SwapEffect = DXGI_SWAP_EFFECT_SEQUENTIAL;
		scDesc.Flags = 0;

		// Get the factory associated with the current device.
		ComPtr<IDXGIDevice1> dxgiDevice;
		ThrowIfFailed(m_d3dDevice.As(&dxgiDevice));

		ComPtr<IDXGIAdapter> dxgiAdapter;
		ThrowIfFailed(dxgiDevice->GetAdapter(&dxgiAdapter));
		
		ComPtr<IDXGIFactory2> dxgiFactory;
		ThrowIfFailed(
			dxgiAdapter->GetParent(
				__uuidof(IDXGIFactory2),
				&dxgiFactory
			)
		);

		// Create the swap chain.
		ThrowIfFailed(
			dxgiFactory->CreateSwapChainForHwnd(
				m_d3dDevice.Get(),
				m_window.lock()->GetHWND(),
				&scDesc,
				nullptr,					// Start windowed
				nullptr,					// Allow on all displays
				&m_swapChain
			)
		);

		CreateD2DWindowSizeDependentResources();

		// Ensure that DXGI does not queue more than one frame at a time. This both reduces latency and
		// ensures that the application will only render after each VSync, minimizing power consumption.
		ThrowIfFailed(
			dxgiDevice->SetMaximumFrameLatency(1)
		);
	}

	// Get the swap chain back buffer.
	ComPtr<ID3D11Texture2D> backBuffer;
	ThrowIfFailed(
		m_swapChain->GetBuffer(
			0,										// The backbuffer is at 0
			__uuidof(ID3D11Texture2D),
			&backBuffer
		)
	);
	SetDebugObjectName(backBuffer.Get(), "D3DBase:Back buffer");

	// Create a render target view of the swap chain back buffer.
	CD3D11_RENDER_TARGET_VIEW_DESC rtvDesc(
		D3D11_RTV_DIMENSION_TEXTURE2D,
		renderTargetViewFormat
	);

	ThrowIfFailed(
		m_d3dDevice->CreateRenderTargetView(
			backBuffer.Get(),
			&rtvDesc,
			&m_finalRenderTargetView
		)
	);
	SetDebugObjectName(m_finalRenderTargetView.Get(), "D3DBase:FinalRenderTargetView");

	// Create the texture that will hold the depth stencil.
	CD3D11_TEXTURE2D_DESC depthStencilDesc(
		dsFormat, 
		bufferWidth,
		bufferHeight,
		1,								// Only one texture
		1,								// Don't mip-map
		dsBindFlags,
		D3D11_USAGE_DEFAULT,
		0,								// CPU access flags
		1,								// No multisampling
		0
	);

	ComPtr<ID3D11Texture2D> depthStencil;
	ThrowIfFailed(
		m_d3dDevice->CreateTexture2D(
			&depthStencilDesc,
			nullptr,
			&depthStencil
		)
	);
	SetDebugObjectName(depthStencil.Get(), "D3DBase:Final Depth Stencil");

	// Create a depth stencil view.
	CD3D11_DEPTH_STENCIL_VIEW_DESC depthStencilViewDesc(
		D3D11_DSV_DIMENSION_TEXTURE2D,
		dsvFormat
	);
	ThrowIfFailed(
		m_d3dDevice->CreateDepthStencilView(
			depthStencil.Get(),
			&depthStencilViewDesc,
			&m_finalDepthStencilView
		)
	);
	SetDebugObjectName(m_finalDepthStencilView.Get(), "D3DBase:FinalDepthStencilView");
	
	// Set the rendering viewport to target the entire window.
	CD3D11_VIEWPORT viewport(
		0.0f,
		0.0f,
		static_cast<float>(bufferWidth),
		static_cast<float>(bufferHeight)
	);

	m_deviceContext->RSSetViewports(1, &viewport);

	m_userAnnotations->EndEvent();
}

void Direct3DBase::UpdateForWindowSizeChange()
{
	XMINT2 newSize = m_window.lock()->GetClientSize();
	ASSERT_MSG(newSize.x > 0 && newSize.y > 0, "Window was resized to a nonpositive or zero size.");

	if (newSize.x != m_renderSize.x ||
		newSize.y != m_renderSize.y)
	{
		ReleaseResources();
		m_deviceContext->Flush();
		CreateWindowSizeDependentResources();
	}
}

void Direct3DBase::Present()
{
	// The application may optionally specify "dirty" or "scroll"
	// rects to improve efficiency in certain scenarios.
	DXGI_PRESENT_PARAMETERS parameters = {0};
	parameters.DirtyRectsCount = 0;
	parameters.pDirtyRects = nullptr;
	parameters.pScrollRect = nullptr;
	parameters.pScrollOffset = nullptr;

	// The first argument instructs DXGI to block until VSync, putting the application
	// to sleep until the next VSync. This ensures we don't waste any cycles rendering
	// frames that will never be displayed to the screen.
	HRESULT hr = m_swapChain->Present1(1, 0, &parameters);

	// Discard the contents of the render target.
	// This is a valid operation only when the existing contents will be entirely
	// overwritten. If dirty or scroll rects are used, this call should be removed.
	m_deviceContext->DiscardView(m_finalRenderTargetView.Get());

	// Discard the contents of the depth stencil.
	m_deviceContext->DiscardView(m_finalDepthStencilView.Get());

	// If the device was removed either by a disconnect or a driver upgrade, we 
	// must recreate all device resources.
	if (hr == DXGI_ERROR_DEVICE_REMOVED) {
		// NOTE: The resource system will need to be refactored to reload resources when this occurs.
		wstring reason = GetErrorMessage(m_d3dDevice->GetDeviceRemovedReason());
		DebugPrint(reason.c_str());
		HandleDeviceLost();
	} else {
		ThrowIfFailed(hr);
	}
}

void Direct3DBase::CheckCapabilities()
{
	// TODO: Implement
}

void Direct3DBase::ReleaseResources()
{
	ID3D11RenderTargetView* nullViews[D3D11_SIMULTANEOUS_RENDER_TARGET_COUNT];
	for(auto& rt : nullViews) {
		rt = nullptr;
	}
	m_deviceContext->OMSetRenderTargets(ARRAYSIZE(nullViews), nullViews, nullptr);

	m_finalRenderTargetView = nullptr;
	m_finalDepthStencilView = nullptr;

	ReleaseD2DWindowSizeDependentResources();
}

void Direct3DBase::CreateD2DWindowSizeDependentResources()
{
#ifdef ENABLE_DIRECT2D
	// Obtain a DXGI surface for D2D
	ComPtr<IDXGISurface> backBuffer;
	ThrowIfFailed(m_swapChain->GetBuffer(0, IID_PPV_ARGS(backBuffer.GetAddressOf())));

	// Create the surface render target
	float dpiX;
	float dpiY;
	m_d2dFactory->GetDesktopDpi(&dpiX, &dpiY);

	D2D1_RENDER_TARGET_PROPERTIES props = D2D1::RenderTargetProperties(
		D2D1_RENDER_TARGET_TYPE_DEFAULT,
		D2D1::PixelFormat(DXGI_FORMAT_UNKNOWN, D2D1_ALPHA_MODE_PREMULTIPLIED),
		dpiX,
		dpiY
		);

	// Create a Direct2D render target which can draw into the surface in the swap chain
	ThrowIfFailed(
		m_d2dFactory->CreateDxgiSurfaceRenderTarget(
		backBuffer.Get(),
		&props,
		&m_d2dBackbufferRT
		)
		);

	// Create the brushes to be used for text.
	ThrowIfFailed(
		m_d2dBackbufferRT->CreateSolidColorBrush(
		D2D1::ColorF(D2D1::ColorF::Black),
		m_blackBrush.GetAddressOf()
		)
		);
	ThrowIfFailed(
		m_d2dBackbufferRT->CreateSolidColorBrush(
		D2D1::ColorF(D2D1::ColorF::White),
		m_whiteBrush.GetAddressOf()
		)
		);

	// Use grayscale text antialiasing for better performance.
	m_d2dBackbufferRT->SetTextAntialiasMode(D2D1_TEXT_ANTIALIAS_MODE_GRAYSCALE);
#endif
}

void Direct3DBase::ReleaseD2DWindowSizeDependentResources()
{
#ifdef ENABLE_DIRECT2D
	m_d2dBackbufferRT = nullptr;
	m_blackBrush = nullptr;
	m_whiteBrush = nullptr;
#endif
}

} // end namespace